# Packet Capture GUI using pktmon
# Requires Administrator privileges to run pktmon

Add-Type -AssemblyName System.Windows.Forms
Add-Type -AssemblyName System.Drawing

# Create default PacketCaptures directory if it doesn't exist
$defaultCaptureDir = "C:\PacketCaptures"
if (-not (Test-Path $defaultCaptureDir)) {
    try {
        New-Item -Path $defaultCaptureDir -ItemType Directory -Force | Out-Null
    }
    catch {
        Write-Warning "Could not create default directory $defaultCaptureDir : $_"
        $defaultCaptureDir = $env:USERPROFILE  # Fallback to user profile
    }
}

# Global variables
$global:generatedFilePath = ""
$global:generatedTxtPath = ""
$global:captureRunning = $false

# Function to generate filename with hostname and timestamp
function Get-CaptureFileName {
    param(
        [string]$SavePath,
        [string]$Extension = "etl"
    )
    
    $hostname = $env:COMPUTERNAME
    $timestamp = Get-Date -Format "yyyyMMdd_HHmmss"
    $filename = "${hostname}_${timestamp}.$Extension"
    
    return Join-Path $SavePath $filename
}

# Function to start packet capture
function Start-PacketCapture {
    param(
        [int]$RuntimeSeconds,
        [string]$SavePath
    )
    
    try {
        # Generate the ETL file path
        $etlFilePath = Get-CaptureFileName -SavePath $SavePath -Extension "etl"
        # Generate the PCAP file path (same name but .pcap extension)
        $global:generatedFilePath = Get-CaptureFileName -SavePath $SavePath -Extension "pcap"
        # Generate the TXT file path (same name but .txt extension)
        $global:generatedTxtPath = Get-CaptureFileName -SavePath $SavePath -Extension "txt"
        
        # Start pktmon capture
        $startResult = & pktmon start --capture --file-name $etlFilePath 2>&1
        
        if ($LASTEXITCODE -ne 0) {
            throw "Failed to start pktmon: $startResult"
        }
        
        $global:captureRunning = $true
        
        # Wait for specified runtime
        Start-Sleep -Seconds $RuntimeSeconds
        
        # Stop pktmon capture
        $stopResult = & pktmon stop 2>&1
        
        if ($LASTEXITCODE -ne 0) {
            Write-Warning "Warning stopping pktmon: $stopResult"
        }
        
        $global:captureRunning = $false
        
        # Convert ETL to PCAP using pktmon
        $convertPcapResult = & pktmon etl2pcap $etlFilePath --out $global:generatedFilePath 2>&1
        $pcapSuccess = $LASTEXITCODE -eq 0
        
        # Convert ETL to TXT using pktmon
        $convertTxtResult = & pktmon etl2txt $etlFilePath --out $global:generatedTxtPath --verbose --timestamp 2>&1
        $txtSuccess = $LASTEXITCODE -eq 0
        
        if (-not $pcapSuccess) {
            Write-Warning "Failed to convert ETL to PCAP: $convertPcapResult"
        } else {

        }
        
        if (-not $txtSuccess) {
            Write-Warning "Failed to convert ETL to TXT: $convertTxtResult"
        } else {

        }
        
        # Only remove ETL file if both conversions succeeded
        if ($pcapSuccess -and $txtSuccess) {
            try {
                Remove-Item $etlFilePath -Force
            } catch {
                Write-Warning "Could not remove temporary ETL file: $etlFilePath"
            }
        } else {
            #Write-Host "ETL file preserved at: $etlFilePath (due to conversion failures)"
            # If PCAP conversion failed, fall back to ETL for Wireshark button
            if (-not $pcapSuccess) {
                $global:generatedFilePath = $etlFilePath
            }
        }
        
        return $true
    }
    catch {
        $global:captureRunning = $false
        Write-Error "Error during packet capture: $_"
        return $false
    }
}

# Function to launch Wireshark with the captured file
function Launch-Wireshark {
    param([string]$FilePath)
    
    try {
        if (-not (Test-Path $FilePath)) {
            [System.Windows.Forms.MessageBox]::Show("Capture file not found: $FilePath", "File Not Found", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Warning)
            return
        }
        
        # Try common Wireshark installation paths
        $wiresharkPaths = @(
            "${env:ProgramFiles}\Wireshark\Wireshark.exe",
            "${env:ProgramFiles(x86)}\Wireshark\Wireshark.exe",
            "C:\Program Files\Wireshark\Wireshark.exe",
            "C:\Program Files (x86)\Wireshark\Wireshark.exe"
        )
        
        $wiresharkPath = $null
        foreach ($path in $wiresharkPaths) {
            if (Test-Path $path) {
                $wiresharkPath = $path
                break
            }
        }
        
        if ($wiresharkPath) {
            Start-Process -FilePath $wiresharkPath -ArgumentList "`"$FilePath`""
        } else {
            [System.Windows.Forms.MessageBox]::Show("Wireshark not found in standard installation paths. Please install Wireshark or launch it manually with the file: $FilePath", "Wireshark Not Found", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Information)
        }
    }
    catch {
        [System.Windows.Forms.MessageBox]::Show("Error launching Wireshark: $_", "Error", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Error)
    }
}

# Function to launch Notepad with the TXT file
function Launch-Notepad {
    param([string]$FilePath)
    
    try {
        if (-not (Test-Path $FilePath)) {
            [System.Windows.Forms.MessageBox]::Show("TXT file not found: $FilePath", "File Not Found", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Warning)
            return
        }
        
        # Launch Notepad with the TXT file
        Start-Process -FilePath "notepad.exe" -ArgumentList "`"$FilePath`""
    }
    catch {
        [System.Windows.Forms.MessageBox]::Show("Error launching Notepad: $_", "Error", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Error)
    }
}

# Create the main form
$form = New-Object System.Windows.Forms.Form
$form.Text = "Packet Capture Tool"
$form.Size = New-Object System.Drawing.Size(500, 340)
$form.StartPosition = "CenterScreen"
$form.FormBorderStyle = "FixedDialog"
$form.MaximizeBox = $false

# Runtime seconds label and textbox
$labelRuntime = New-Object System.Windows.Forms.Label
$labelRuntime.Location = New-Object System.Drawing.Point(20, 20)
$labelRuntime.Size = New-Object System.Drawing.Size(150, 20)
$labelRuntime.Text = "Capture Duration (seconds):"
$form.Controls.Add($labelRuntime)

$textboxRuntime = New-Object System.Windows.Forms.TextBox
$textboxRuntime.Location = New-Object System.Drawing.Point(180, 18)
$textboxRuntime.Size = New-Object System.Drawing.Size(100, 20)
$textboxRuntime.Text = "30"
$form.Controls.Add($textboxRuntime)

# Save location label and textbox
$labelSavePath = New-Object System.Windows.Forms.Label
$labelSavePath.Location = New-Object System.Drawing.Point(20, 60)
$labelSavePath.Size = New-Object System.Drawing.Size(100, 20)
$labelSavePath.Text = "Save Location:"
$form.Controls.Add($labelSavePath)

$textboxSavePath = New-Object System.Windows.Forms.TextBox
$textboxSavePath.Location = New-Object System.Drawing.Point(20, 80)
$textboxSavePath.Size = New-Object System.Drawing.Size(350, 20)
$textboxSavePath.Text = $defaultCaptureDir
$form.Controls.Add($textboxSavePath)

# Browse button for save location
$buttonBrowse = New-Object System.Windows.Forms.Button
$buttonBrowse.Location = New-Object System.Drawing.Point(380, 78)
$buttonBrowse.Size = New-Object System.Drawing.Size(80, 25)
$buttonBrowse.Text = "Browse..."
$buttonBrowse.Add_Click({
    $folderDialog = New-Object System.Windows.Forms.FolderBrowserDialog
    $folderDialog.SelectedPath = $textboxSavePath.Text
    $folderDialog.Description = "Select folder to save packet capture files"
    
    if ($folderDialog.ShowDialog() -eq [System.Windows.Forms.DialogResult]::OK) {
        $textboxSavePath.Text = $folderDialog.SelectedPath
    }
})
$form.Controls.Add($buttonBrowse)

# Status label
$labelStatus = New-Object System.Windows.Forms.Label
$labelStatus.Location = New-Object System.Drawing.Point(20, 120)
$labelStatus.Size = New-Object System.Drawing.Size(440, 20)
$labelStatus.Text = "Ready to capture packets"
$labelStatus.ForeColor = [System.Drawing.Color]::Green
$form.Controls.Add($labelStatus)

# Current file label
$labelCurrentFile = New-Object System.Windows.Forms.Label
$labelCurrentFile.Location = New-Object System.Drawing.Point(20, 150)
$labelCurrentFile.Size = New-Object System.Drawing.Size(440, 40)
$labelCurrentFile.Text = "No capture file generated yet"
$labelCurrentFile.ForeColor = [System.Drawing.Color]::Blue
$form.Controls.Add($labelCurrentFile)

# Start capture button
$buttonStartCapture = New-Object System.Windows.Forms.Button
$buttonStartCapture.Location = New-Object System.Drawing.Point(20, 220)
$buttonStartCapture.Size = New-Object System.Drawing.Size(140, 40)
$buttonStartCapture.Text = "Start Packet Capture"
$buttonStartCapture.BackColor = [System.Drawing.Color]::LightGreen
$buttonStartCapture.Add_Click({
    try {
        # Validate inputs
        $runtime = 0
        if (-not [int]::TryParse($textboxRuntime.Text, [ref]$runtime) -or $runtime -le 0) {
            [System.Windows.Forms.MessageBox]::Show("Please enter a valid positive number for capture duration.", "Invalid Input", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Warning)
            return
        }
        
        if (-not (Test-Path $textboxSavePath.Text)) {
            [System.Windows.Forms.MessageBox]::Show("The specified save path does not exist.", "Invalid Path", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Warning)
            return
        }
        
        # Disable button and update status
        $buttonStartCapture.Enabled = $false
        $buttonLaunchWireshark.Enabled = $false
        $buttonLaunchNotepad.Enabled = $false
        $labelStatus.Text = "Capturing packets and converting to PCAP/TXT... Please wait"
        $labelStatus.ForeColor = [System.Drawing.Color]::Orange
        $form.Refresh()
        
        # Start capture in background
        $result = Start-PacketCapture -RuntimeSeconds $runtime -SavePath $textboxSavePath.Text
        
        if ($result) {
            $labelStatus.Text = "Packet capture and file conversions completed successfully!"
            $labelStatus.ForeColor = [System.Drawing.Color]::Green
            $labelCurrentFile.Text = "Generated files: $global:generatedFilePath`nTXT file: $global:generatedTxtPath"
            # Enable buttons based on file availability
            if (Test-Path $global:generatedFilePath) {
                $buttonLaunchWireshark.Enabled = $true
            }
            if (Test-Path $global:generatedTxtPath) {
                $buttonLaunchNotepad.Enabled = $true
            }
        } else {
            $labelStatus.Text = "Packet capture or conversion failed. Check console for details."
            $labelStatus.ForeColor = [System.Drawing.Color]::Red
        }
        
        $buttonStartCapture.Enabled = $true
    }
    catch {
        $labelStatus.Text = "Error: $_"
        $labelStatus.ForeColor = [System.Drawing.Color]::Red
        $buttonStartCapture.Enabled = $true
        [System.Windows.Forms.MessageBox]::Show("An error occurred: $_", "Error", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Error)
    }
})
$form.Controls.Add($buttonStartCapture)

# Launch Wireshark button
$buttonLaunchWireshark = New-Object System.Windows.Forms.Button
$buttonLaunchWireshark.Location = New-Object System.Drawing.Point(170, 220)
$buttonLaunchWireshark.Size = New-Object System.Drawing.Size(140, 40)
$buttonLaunchWireshark.Text = "Launch Wireshark"
$buttonLaunchWireshark.BackColor = [System.Drawing.Color]::LightBlue
$buttonLaunchWireshark.Enabled = $false
$buttonLaunchWireshark.Add_Click({
    if ($global:generatedFilePath -and (Test-Path $global:generatedFilePath)) {
        Launch-Wireshark -FilePath $global:generatedFilePath
    } else {
        [System.Windows.Forms.MessageBox]::Show("No capture file available. Please run a packet capture first.", "No File", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Information)
    }
})
$form.Controls.Add($buttonLaunchWireshark)

# Launch Notepad button
$buttonLaunchNotepad = New-Object System.Windows.Forms.Button
$buttonLaunchNotepad.Location = New-Object System.Drawing.Point(320, 220)
$buttonLaunchNotepad.Size = New-Object System.Drawing.Size(140, 40)
$buttonLaunchNotepad.Text = "Launch Notepad"
$buttonLaunchNotepad.BackColor = [System.Drawing.Color]::LightYellow
$buttonLaunchNotepad.Enabled = $false
$buttonLaunchNotepad.Add_Click({
    if ($global:generatedTxtPath -and (Test-Path $global:generatedTxtPath)) {
        Launch-Notepad -FilePath $global:generatedTxtPath
    } else {
        [System.Windows.Forms.MessageBox]::Show("No TXT file available. Please run a packet capture first.", "No File", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Information)
    }
})
$form.Controls.Add($buttonLaunchNotepad)

# Admin check warning
$isAdmin = ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)

if (-not $isAdmin) {
    $labelAdmin = New-Object System.Windows.Forms.Label
$labelAdmin.Location = New-Object System.Drawing.Point(20, 270)
    $labelAdmin.Size = New-Object System.Drawing.Size(440, 20)
    $labelAdmin.Text = "WARNING: This script requires Administrator privileges to run pktmon!"
    $labelAdmin.ForeColor = [System.Drawing.Color]::Red
    $labelAdmin.Font = New-Object System.Drawing.Font("Microsoft Sans Serif", 9, [System.Drawing.FontStyle]::Bold)
    $form.Controls.Add($labelAdmin)
}

# Show the form
[System.Windows.Forms.Application]::EnableVisualStyles()
$form.ShowDialog() | Out-Null